import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:zcart_delivery/config/config.dart';
import 'package:zcart_delivery/helper/constants.dart';
import 'package:zcart_delivery/providers/auth_provider.dart';
import 'package:zcart_delivery/providers/order_provider.dart';
import 'package:zcart_delivery/translations/locale_keys.g.dart';
import 'package:zcart_delivery/views/custom/information_card_icon.dart';
import 'package:zcart_delivery/views/orders/orders_page.dart';
import 'package:easy_localization/easy_localization.dart';

class DashboardPage extends ConsumerWidget {
  const DashboardPage({super.key});

  @override
  Widget build(BuildContext context, ref) {
    return Scaffold(
      appBar: _appbar(ref),
      body: _body(context, ref),
    );
  }

  Widget _body(BuildContext context, WidgetRef ref) {
    final userRef = ref.watch(userProvider);

    final ordersRef = ref.watch(ordersProvider);
    return RefreshIndicator(
      onRefresh: () async {
        _refreshData(ref);
      },
      child: SingleChildScrollView(
        child: Padding(
          padding: const EdgeInsets.all(defaultPadding * 1.5),
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.stretch,
            children: [
              userRef.when(
                data: (user) {
                  return user != null
                      ? Row(
                          children: [
                            ClipOval(
                              child: user.avatar != null &&
                                      user.avatar!.isNotEmpty
                                  ? Image.network(
                                      user.avatar!,
                                      width: 64,
                                      height: 64,
                                      fit: BoxFit.cover,
                                      errorBuilder:
                                          (context, error, stackTrace) {
                                        return CircleAvatar(
                                          backgroundColor:
                                              MyConfig.primaryColor,
                                          radius: 50,
                                          child: const Icon(
                                            Icons.person,
                                            color: Colors.white,
                                            size: 50,
                                          ),
                                        );
                                      },
                                    )
                                  : CircleAvatar(
                                      backgroundColor: MyConfig.primaryColor,
                                      radius: 50,
                                      child: const Icon(
                                        Icons.person,
                                        color: Colors.white,
                                        size: 50,
                                      ),
                                    ),
                            ),
                            const SizedBox(width: defaultPadding),
                            Expanded(
                              child: Column(
                                crossAxisAlignment: CrossAxisAlignment.start,
                                children: [
                                  Text(
                                    '${user.firstName.isEmpty ? 'No Name' : '${user.firstName} '}${user.lastName.isEmpty ? '' : user.lastName}',
                                    style: const TextStyle(
                                      fontSize: 18,
                                      fontWeight: FontWeight.bold,
                                    ),
                                  ),
                                  Text(
                                    user.email,
                                    style: const TextStyle(
                                      fontSize: 14,
                                      fontWeight: FontWeight.w400,
                                    ),
                                  ),
                                ],
                              ),
                            ),
                            // Rating
                            Row(
                              children: [
                                const Icon(Icons.star,
                                    color: Colors.orange, size: 18),
                                const SizedBox(width: 2),
                                Text(
                                  user.avgRating.toStringAsFixed(1),
                                  style: const TextStyle(
                                      fontWeight: FontWeight.bold),
                                ),
                              ],
                            ),
                          ],
                        )
                      : const SizedBox();
                },
                error: (error, stackTrace) => const SizedBox(),
                loading: () => const SizedBox(),
              ),
              const SizedBox(height: defaultPadding * 2),
              ordersRef.when(
                data: (data) {
                  if (data == null) {
                    return const SizedBox();
                  } else {
                    final newData = data
                        .where((element) =>
                            element.orderStatus != 'delivered'.toUpperCase())
                        .toList();

                    return InformationCardWithIcon(
                      icon: Icons.shopping_bag,
                      iconColor: Colors.orange,
                      number: newData.length.toString(),
                      title: LocaleKeys.pending_delivery.tr(),
                      onTap: () {
                        Navigator.push(
                          context,
                          MaterialPageRoute(
                            builder: (context) =>
                                const OrdersPage(isOnlyPending: true),
                          ),
                        );
                      },
                    );
                  }
                },
                error: (error, stackTrace) {
                  return const SizedBox();
                },
                loading: () => const InfoCardShimmer(),
              ),
              const SizedBox(height: defaultPadding),
              ordersRef.when(
                data: (data) {
                  return data != null
                      ? InformationCardWithIcon(
                          icon: Icons.shopping_basket,
                          iconColor: MyConfig.primaryColor,
                          number: data.length.toString(),
                          title: LocaleKeys.all_orders.tr(),
                          onTap: () {
                            Navigator.push(
                              context,
                              MaterialPageRoute(
                                builder: (context) => const OrdersPage(),
                              ),
                            );
                          },
                        )
                      : const SizedBox();
                },
                error: (error, stackTrace) => const SizedBox(),
                loading: () => const InfoCardShimmer(),
              ),
              const SizedBox(height: defaultPadding),
              ordersRef.when(
                data: (data) {
                  if (data == null) {
                    return const SizedBox();
                  }
                  final newData = data
                      .where((element) =>
                          element.orderStatus == 'delivered'.toUpperCase())
                      .toList();
                  return InformationCardWithIcon(
                    icon: CupertinoIcons.rectangle_grid_2x2_fill,
                    iconColor: Colors.green,
                    number: newData.length.toString(),
                    title: LocaleKeys.completed_delivery.tr(),
                    onTap: () {
                      Navigator.push(
                        context,
                        MaterialPageRoute(
                          builder: (context) =>
                              const OrdersPage(isOnlyDelivered: true),
                        ),
                      );
                    },
                  );
                },
                error: (error, stackTrace) => const SizedBox(),
                loading: () => const InfoCardShimmer(),
              ),
              const SizedBox(height: defaultPadding),
            ],
          ),
        ),
      ),
    );
  }

  AppBar _appbar(WidgetRef ref) {
    return AppBar(
      title: Text(LocaleKeys.dashboard.tr()),
      actions: [
        IconButton(
          tooltip: LocaleKeys.refresh.tr(),
          icon: const Icon(Icons.sync),
          onPressed: () {
            _refreshData(ref);
          },
        ),
      ],
    );
  }
}

void _refreshData(WidgetRef ref) {
  ref.invalidate(ordersProvider);
}
